<?php

namespace App\Http\Controllers\Auth;

use App\Company;
use App\User;
use \Illuminate\Http\Request;
use Validator;
use App\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\ThrottlesLogins;
use Illuminate\Foundation\Auth\AuthenticatesAndRegistersUsers;

class AuthController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Registration & Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users, as well as the
    | authentication of existing users. By default, this controller uses
    | a simple trait to add these behaviors. Why don't you explore it?
    |
    */
    protected $redirectPath = '/dashboard';
    protected $loginPath = '/';
    protected $username = 'company_email';
    use AuthenticatesAndRegistersUsers, ThrottlesLogins;


    public function __construct()
    {
        $this->middleware('guest', ['except' => 'getLogout']);
    }

    public function getLogout()
    {
        \Auth::logout();
        $path = '/';
        if (\Session::has('AdminMode')) {
            $path = '/admin';
            \Session::forget('AdminMode');
            \Session::save();
        }
        if (\Session::has('SuperMode')) {
            $path = '/super';
            \Session::forget('SuperMode');
            \Session::save();
        }
        if (\Session::has('company')) {
            \Session::forget('company');
            \Session::save();
        }
        return redirect($path);
    }

    public function getAdminLogin()
    {
        return view('auth.login')->withAdminLogin(true);
    }

    public function getSuperLogin()
    {
        return view('auth.login')->withSuperLogin(true);
    }

    public function postAdminLogin(Request $request)
    {

        $this->validate($request, [
            $this->loginUsername() => 'required', 'password' => 'required',
        ]);

        // If the class is using the ThrottlesLogins trait, we can automatically throttle
        // the login attempts for this application. We'll key this by the username and
        // the IP address of the client making these requests into this application.
        $throttles = $this->isUsingThrottlesLoginsTrait();

        if ($throttles && $this->hasTooManyLoginAttempts($request)) {
            return $this->sendLockoutResponse($request);
        }

        $credentials = $this->getCredentials($request);
        $errors = [];
        if (\Auth::attempt($credentials, $request->has('remember'))) {
            if (!in_array(\Auth::user()->getRole(), ['HRAdmin', 'CompanyAdmin'])) {
                $errors[] = "You don't have the required privileges to access this section";
            }
            if (!Company::isActive(\Auth::user()->company_id)) {
                $errors[] = "Subscription ended, Kindly contact our customer support";
            }
            if (count($errors) > 0) {
                \Auth::logout();
                return redirect()->back()->withErrors($errors);
            }
            \Session::put('AdminMode', true);
            \Session::put('settings',Company::getSettings(\Auth::user()->company_id));
            \Session::save();
            return $this->handleUserWasAuthenticated($request, $throttles);
        }

        // If the login attempt was unsuccessful we will increment the number of attempts
        // to login and redirect the user back to the login form. Of course, when this
        // user surpasses their maximum number of attempts they will get locked out.
        if ($throttles) {
            $this->incrementLoginAttempts($request);
        }

        return redirect('/admin')
            ->withInput($request->only($this->loginUsername(), 'remember'))
            ->withErrors([
                $this->loginUsername() => $this->getFailedLoginMessage(),
            ]);
    }

    public function getAccessedAs($domain,$access, Request $request){

        $company = Company::whereAccessedAs($access)->first();
        $company->accessed_as = null;
        $company->save();
        $user = User::find($company->user_id);
        \Auth::login($user);
        \Session::forget('SuperMode');
        \Session::put('AdminMode', true);
        \Session::put('settings',Company::getSettings($user->company_id));
        \Session::save();
        $errors =[];
        if (!Company::isActive(\Auth::user()->company_id))
            $errors[] = "Subscription ended for this company, reactivate and try to access again";
        if (count($errors) > 0) {
            \Auth::logout();
            return redirect($this->loginPath())->withErrors($errors);
        }    
        return redirect(action('UserController@getDashboard'));
    }
    public function postLogin($company,Request $request)
    {
        $this->validate($request, [
            'company_email' => 'required', 'password' => 'required',
        ]);

        // If the class is using the ThrottlesLogins trait, we can automatically throttle
        // the login attempts for this application. We'll key this by the username and
        // the IP address of the client making these requests into this application.
        $throttles = $this->isUsingThrottlesLoginsTrait();

        if ($throttles && $this->hasTooManyLoginAttempts($request)) {
            return $this->sendLockoutResponse($request);
        }

        $credentials = $request->only(['company_email', 'password']);
        $errors = [];
        if (\Auth::attempt($credentials)) {
            if (!Company::isActive(\Auth::user()->company_id))
                $errors[] = "Subscription ended, Kindly contact your HR Department";
            if (count($errors) > 0) {
                \Auth::logout();
                return redirect()->back()->withErrors($errors);
            }
            return redirect(action('UserController@getDashboard'));
        }

        // If the login attempt was unsuccessful we will increment the number of attempts
        // to login and redirect the user back to the login form. Of course, when this
        // user surpasses their maximum number of attempts they will get locked out.
        if ($throttles) {
            $this->incrementLoginAttempts($request);
        }

        return redirect($this->loginPath())
            ->withInput($request->only($this->loginUsername(), 'remember'))
            ->withErrors([
                $this->loginUsername() => $this->getFailedLoginMessage(),
            ]);
    }

    public function postSuperLogin(Request $request)
    {

        $this->validate($request, [
            'company_email' => 'required', 'password' => 'required',
        ]);


        // If the class is using the ThrottlesLogins trait, we can automatically throttle
        // the login attempts for this application. We'll key this by the username and
        // the IP address of the client making these requests into this application.
        $throttles = $this->isUsingThrottlesLoginsTrait();

        if ($throttles && $this->hasTooManyLoginAttempts($request)) {
            return $this->sendLockoutResponse($request);
        }

        $credentials = $request->only(['company_email', 'password']);
        $errors = [];
        if (\Auth::attempt($credentials)) {
            if (\Auth::user()->getRole() != 'SuperAdmin') {
                $errors[] = "You don't have the required privileges to access this section";
            }
            if (count($errors) > 0) {
                \Auth::logout();
                return redirect()->back()->withErrors($errors);
            }
            \Session::put('SuperMode', true);
            \Session::forget('company');
            \Session::save();
            if ($throttles) {
                $this->clearLoginAttempts($request);
            }
            return redirect()->action('CompaniesController@getList');
        }

        // If the login attempt was unsuccessful we will increment the number of attempts
        // to login and redirect the user back to the login form. Of course, when this
        // user surpasses their maximum number of attempts they will get locked out.
        if ($throttles) {
            $this->incrementLoginAttempts($request);
        }

        return redirect('/super')
            ->withInput($request->only($this->loginUsername(), 'remember'))
            ->withErrors([
                $this->loginUsername() => $this->getFailedLoginMessage(),
            ]);
    }

    public function getResetPassword()
    {
        return view('auth.reset');
    }

    public function postResetPassword(Request $request)
    {
        $this->validate($request, ['company_email' => 'required|email|exists:users,company_email']);
        if (User::resetPasswordByEmail($request->company_email)) {
            \Session::put('system_message', 'Your new password was sent to ' . $request->company_email);
            \Session::save();
            return redirect(action('Auth\AuthController@getLogin'));
        }
        return redirect()->back()->withErrors(['Unable to reset your password']);
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'name' => 'required|max:255',
            'company_email' => 'required|email|max:255|unique:users',
            'password' => 'required',
        ]);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array $data
     * @return User
     */
    protected function create(array $data)
    {
        return User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => bcrypt($data['password']),
        ]);
    }

    public function getLogin($domain)
    {
        if (Company::checkDomainExists($domain)) {
            \Session::put('domain',$domain);
            \Session::save();
            if (view()->exists('auth.authenticate')) {
                return view('auth.authenticate');
            }
            return view('auth.login');
        }
        return view('errors.404');
    }
}
